<?php
/*
Plugin Name: Category Access
Plugin URI: http://www.coppit.org/code/
Description: Protects categories of posts from specific users.
Version: 0.5.3
Author: David Coppit
Author URI: http://www.coppit.org
*/

include_once('category-access-options.php');
include_once('category-access-user-edit.php');

// --------------------------------------------------------------------

global $category_access_default_private_message;
$category_access_default_private_message =
	'Sorry, you do not have sufficient privileges to view this post.';

class category_access {

function filter_category_list_item($text,$category = null)
{
	// In case this filter was called with the whole list. Not sure what the
	// right way is to detect this case.
	if (strpos($text,"\n") || strpos($text,"<\/a>"))
		return $text;

	// If Category_Access_show_private_categories is false, then all of the
	// items we will be given will be valid, since filter_category_list would
	// have already filtered out all the illicit ones. We can also abort early
	// if we don't plan to make any changes.
	if (!get_option('Category_Access_show_private_categories') ||
			!get_option('Category_Access_show_padlock_on_private_categories'))
		return $text;


    $category_id = $category->id;

    // If we're unlucky we won't be given the category and will have to guess
	// it from the name
	if ($category_id == null) {
		foreach (get_all_category_ids() as $possible_category_id) {
			if (get_catname($possible_category_id) == $text) {
				$category_id = $possible_category_id;
				break;
			}
		}
	}

	global $current_user;

	if (category_access::user_can_access_category($category_id, $current_user))
		return $text;


	# Make the changes
	$site_root = parse_url(get_settings('siteurl'));
	$site_root = trailingslashit($site_root['path']);

	return $text . ' <img src=\'' . $site_root .
		'wp-content/plugins/category-access/padlock.gif\' ' .
		'height=\'10\' width=\'8\' valign=\'middle\' border=\'0\'/>';
}

// --------------------------------------------------------------------

function filter_category_list($exclusions)
{
	global $current_user;

	if (get_option('Category_Access_show_private_categories'))
		return $exclusions;

	foreach (get_all_category_ids() as $category_id) {
		if (!category_access::user_can_access_category($category_id, $current_user))
			$exclusions .= " AND cat_ID <> $category_id ";
	}

	return $exclusions;
}

// --------------------------------------------------------------------

function post_should_be_hidden($postid)
{
	$post = get_post($postid);

	if ($post->post_status == 'static')
		return false;

	if (!isset($postid))
		return true;

	global $current_user;

	$post_categories = wp_get_post_cats(1, $postid);

	if (get_option('Category_Access_show_if_any_category_visible')) {
		foreach ($post_categories as $post_category_id)
			if (category_access::user_can_access_category(
					$post_category_id, $current_user))
				return false;

		return true;
	} else {
		foreach ($post_categories as $post_category_id)
			if (!category_access::user_can_access_category(
					$post_category_id, $current_user))
				return true;

		return false;
	}
}

// --------------------------------------------------------------------

function user_can_access_category ($cat_id,$user)
{
	do {
		if (!category_access::get_category_access_for_user($cat_id, $user))
			return false;

		$this_category = get_category($cat_id);
		$cat_id = $this_category->category_parent;
	} while ($cat_id != 0);

	return true;
}

// --------------------------------------------------------------------

function filter_title($text)
{
	global $post;

	if (category_access::post_should_be_hidden($post->ID))
		return category_access::get_private_message();

	return $text;
}

// --------------------------------------------------------------------

function get_private_message() {
	$message = get_option("Category_Access_private_message");

	if ($message == null) {
		global $category_access_default_private_message;
		$message = $category_access_default_private_message;
	}

	return $message;
}

// --------------------------------------------------------------------

function get_category_access_for_user($category_id, $user) {
	if ($user->has_cap('manage_categories'))
		return true;

	$user_id = $user->id;

	if ($user_id == 0)
		return get_option("Category_Access_cat_${category_id}_anonymous");

	$visible = get_option("Category_Access_cat_${category_id}_user_${user_id}");

	if ($visible === false)
		$visible = get_option("Category_Access_cat_${category_id}_default");

	return $visible;
}

// --------------------------------------------------------------------

function set_category_access_for_user($category_id, $user_id, $value) {
	if ($value)
		update_option("Category_Access_cat_${category_id}_user_${user_id}", '1');
	else
		update_option("Category_Access_cat_${category_id}_user_${user_id}", '0');
}

// --------------------------------------------------------------------

function filter_posts($allposts)
{
	global $feed;

	if ( (empty($feed) && get_option('Category_Access_show_private_message')) ||
			!is_array($allposts) )
		return $allposts;

	foreach ($allposts as $post)
		if (!category_access::post_should_be_hidden($post->ID))
			$goodposts[] = $post;

	return $goodposts;
}

// --------------------------------------------------------------------

function post_padlock($text)
{
	global $post;

	if (!category_access::post_should_be_hidden($post->ID))
		return $text;

	if (get_option('Category_Access_show_padlock_on_private_posts'))
	{
		$site_root = parse_url(get_settings('siteurl'));
		$site_root = trailingslashit($site_root['path']);

		$text = '<img src=\'' . $site_root .
			'wp-content/plugins/category-access/padlock.gif\' ' .
			'valign=\'middle\' border=\'0\'/>' . $text;
	}

	return $text;
}

// --------------------------------------------------------------------

function hide_text($text)
{
	global $post;

	if (category_access::post_should_be_hidden($post->ID))
		$text = '';

	return $text;
}

// --------------------------------------------------------------------

// Used during development to dump a data structure to html
function print_html_data($data) {
	$string = htmlspecialchars(print_r($data,1));
	$string = preg_replace("/\n/", "<br>\n", $string);
	$string = preg_replace("/ /", "&nbsp;", $string);

	print $string;
}

// --------------------------------------------------------------------

function backtrace()
{
   $output = "<div style='text-align: left; font-family: monospace;'>\n";
   $output .= "<b>Backtrace:</b><br />\n";
   $backtrace = debug_backtrace();

   foreach ($backtrace as $bt) {
       $args = '';
       foreach ($bt['args'] as $a) {
           if (!empty($args)) {
               $args .= ', ';
           }
           switch (gettype($a)) {
           case 'integer':
           case 'double':
               $args .= $a;
               break;
           case 'string':
               $a = htmlspecialchars(substr($a, 0, 64)).((strlen($a) > 64) ? '...' : '');
               $args .= "\"$a\"";
               break;
           case 'array':
               $args .= 'Array('.count($a).')';
               break;
           case 'object':
               $args .= 'Object('.get_class($a).')';
               break;
           case 'resource':
               $args .= 'Resource('.strstr($a, '#').')';
               break;
           case 'boolean':
               $args .= $a ? 'True' : 'False';
               break;
           case 'NULL':
               $args .= 'Null';
               break;
           default:
               $args .= 'Unknown';
           }
       }
       $output .= "<br />\n";
       $output .= "<b>file:</b> {$bt['line']} - {$bt['file']}<br />\n";
       $output .= "<b>call:</b> {$bt['class']}{$bt['type']}{$bt['function']}($args)<br />\n";
   }
   $output .= "</div>\n";
   return $output;
}
// ####################################################################

function next_post_link($format='%link &raquo;', $link='%title', $in_same_cat = false, $excluded_categories = '') {
	// vvvvvvvvvvvvvvvvvvvvvvvvvvvvv Category Access custom exclusions
	global $current_user;

	foreach (get_all_category_ids() as $category_id) {
		if (!category_access::get_category_access_for_user($category_id, $current_user)) {
			if (empty($excluded_categories))
			  	$excluded_categories = $category_id;
			else
				$excluded_categories .= " and $category_id";
		}
	}
	// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^

	// vvvvvvvvvvvvvvvvvvvvvvvvvvvvv Call our patched version
	// See http://trac.wordpress.org/ticket/2215
	$post = category_access::get_next_post($in_same_cat, $excluded_categories);

	if ( !$post ) {
		return;
	}

	$title = apply_filters('the_title', $post->post_title, $post);
	$string = '<a href="'.get_permalink($post->ID).'">';
	$link = str_replace('%title', $title, $link);
	$link = $string . $link . '</a>';
	$format = str_replace('%link', $link, $format);

	echo $format;	    
}

// --------------------------------------------------------------------

function get_next_post($in_same_cat = false, $excluded_categories = '') {
	global $post, $wpdb;

	if( !is_single() || is_attachment() )
		return null;

	$current_post_date = $post->post_date;
	
	$join = '';
	if ( $in_same_cat ) {
		$join = " INNER JOIN $wpdb->post2cat ON $wpdb->posts.ID= $wpdb->post2cat.post_id ";
		$cat_array = get_the_category($post->ID);
		$join .= ' AND (category_id = ' . intval($cat_array[0]->cat_ID);
		for ( $i = 1; $i < (count($cat_array)); $i++ ) {
			$join .= ' OR category_id = ' . intval($cat_array[$i]->cat_ID);
		}
		$join .= ')'; 
	}

	$posts_in_ex_cats_sql = '';
	if ( !empty($excluded_categories) ) {
		$blah = explode(' and ', $excluded_categories);
		$sql_cat_ids = '';
		foreach ( $blah as $category ) {
			$category = intval($category);
			$sql_cat_ids .= " OR pc.category_ID='$category'";
		}
		$posts_in_ex_cats = $wpdb->get_col("SELECT p.ID from $wpdb->posts p LEFT JOIN $wpdb->post2cat pc ON pc.post_id=p.ID WHERE 1=0 $sql_cat_ids GROUP BY p.ID");

		if ( !empty($posts_in_ex_cats) )
			$posts_in_ex_cats_sql = 'AND ID NOT IN (' . implode($posts_in_ex_cats, ',') . ')';
	}

	$now = current_time('mysql');
	
	return @$wpdb->get_row("SELECT ID,post_title FROM $wpdb->posts $join WHERE post_date > '$current_post_date' AND post_date < '$now' AND post_status = 'publish' $posts_in_ex_cats_sql AND ID != $post->ID ORDER BY post_date ASC LIMIT 1");
}

// --------------------------------------------------------------------

function previous_post_link($format='%link &raquo;', $link='%title', $in_same_cat = false, $excluded_categories = '') {
	// vvvvvvvvvvvvvvvvvvvvvvvvvvvvv Category Access custom exclusions
	global $current_user;

	foreach (get_all_category_ids() as $category_id) {
		if (!category_access::get_category_access_for_user($category_id, $current_user)) {
			if (empty($excluded_categories))
			  	$excluded_categories = $category_id;
			else
				$excluded_categories .= " and $category_id";
		}
	}
	// ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^

	if ( is_attachment() )
		$post = & get_post($GLOBALS['post']->post_parent);
	else
		// vvvvvvvvvvvvvvvvvvvvvvvvvvvvv Call our patched version
		// See http://trac.wordpress.org/ticket/2215
		$post = category_access::get_previous_post($in_same_cat, $excluded_categories);

	if ( !$post )
		return;

	$title = apply_filters('the_title', $post->post_title, $post);
	$string = '<a href="'.get_permalink($post->ID).'">';
	$link = str_replace('%title', $title, $link);
	$link = $pre . $string . $link . '</a>';

	$format = str_replace('%link', $link, $format);

	echo $format;	    
}

// --------------------------------------------------------------------

function get_previous_post($in_same_cat = false, $excluded_categories = '') {
	global $post, $wpdb;

	if( !is_single() || is_attachment() )
		return null;

	$current_post_date = $post->post_date;

	$join = '';
	if ( $in_same_cat ) {
		$join = " INNER JOIN $wpdb->post2cat ON $wpdb->posts.ID= $wpdb->post2cat.post_id ";
		$cat_array = get_the_category($post->ID);
		$join .= ' AND (category_id = ' . intval($cat_array[0]->cat_ID);
		for ( $i = 1; $i < (count($cat_array)); $i++ ) {
			$join .= ' OR category_id = ' . intval($cat_array[$i]->cat_ID);
		}
		$join .= ')'; 
	}

	$posts_in_ex_cats_sql = '';
	if ( !empty($excluded_categories) ) {
		$blah = explode(' and ', $excluded_categories);
		$sql_cat_ids = '';
		foreach ( $blah as $category ) {
			$category = intval($category);
			$sql_cat_ids .= " OR pc.category_ID='$category'";
		}
		$posts_in_ex_cats = $wpdb->get_col("SELECT p.ID from $wpdb->posts p LEFT JOIN $wpdb->post2cat pc ON pc.post_id=p.ID WHERE 1=0 $sql_cat_ids GROUP BY p.ID");
		if ( !empty($posts_in_ex_cats) )
			$posts_in_ex_cats_sql = 'AND ID NOT IN (' . implode($posts_in_ex_cats, ',') . ')';
	}

	return @$wpdb->get_row("SELECT ID, post_title FROM $wpdb->posts $join WHERE post_date < '$current_post_date' AND post_status = 'publish' $posts_in_ex_cats_sql ORDER BY post_date DESC LIMIT 1");
}

// --------------------------------------------------------------------

function get_valid_categories() {
	global $current_user;

	foreach (get_all_category_ids() as $category_id) {
		if (!category_access::get_category_access_for_user($category_id, $current_user))
			$exclusions[] = $category_id;
	}
	return $exclusions;
}

// --------------------------------------------------------------------

// If the user saves a post in a category or categories from which they are
// restricted, remove the post from the restricted category(ies).  If there
// are no categories left, save it as Uncategorized with status 'Saved'.
function verify_category($post_ID) {
	global $wpdb;
	
	$postcats = $wpdb->get_col("SELECT category_id FROM $wpdb->post2cat WHERE post_id = $post_ID ORDER BY category_id");
	$exclusions = category_access::get_valid_categories();

	if (count($exclusions)) {
		$exclusions = implode(", ", $exclusions);
		$wpdb->query("DELETE FROM $wpdb->post2cat WHERE post_id = $post_ID AND category_id IN ($exclusions)");
		$good_cats = $wpdb->get_var("SELECT COUNT(*) FROM $wpdb->post2cat WHERE post_id = $post_ID");

		if (0 == $good_cats) {
			$wpdb->query("INSERT INTO $wpdb->post2cat (`post_id`, `category_id`) VALUES ($post_ID, 1)");
			$wpdb->query("UPDATE $wpdb->posts SET post_status = 'draft' WHERE ID = $post_ID");
		}
	}
}

}

// --------------------------------------------------------------------

add_filter('list_cats', array('category_access','filter_category_list_item'));
add_filter('list_cats_exclusions', array('category_access','filter_category_list'));
add_filter('the_title', array('category_access','filter_title'));
add_filter('single_post_title', array('category_access','filter_title'));
add_filter('the_title_rss', array('category_access','filter_title'));
add_filter('the_title', array('category_access','post_padlock'));
add_filter('the_content', array('category_access','hide_text'));
add_filter('the_excerpt', array('category_access','hide_text'));
add_filter('comment_url', array('category_access','hide_text'));
add_filter('comment_excerpt', array('category_access','hide_text'));
add_filter('comment_email', array('category_access','hide_text'));
add_filter('comment_author', array('category_access','hide_text'));
add_filter('comment_text', array('category_access','hide_text'));
add_filter('the_posts', array('category_access','filter_posts'));
add_action('save_post', array('category_access','verify_category'));

?>
